// Mermaid 初始化和渲染脚本
(async function () {
	'use strict';

	// 获取全局 vscode API（由 webview-common.js 提供）
	const getVsCodeApi = () => window.vscodeApi;

	// 国际化字符串 - 这些将由扩展主进程传递过来
	let i18nStrings = {};

	// 国际化函数
	function t(key, ...args) {
		let text = i18nStrings[key] || key;
		// 简单的参数替换，支持 {0}, {1} 等占位符
		if (args.length > 0) {
			args.forEach((arg, index) => {
				text = text.replace(new RegExp(`\\{${index}\\}`, 'g'), arg);
			});
		}
		return text;
	}

	// 监听来自后端的消息
	window.addEventListener('message', event => {
		const message = event.data;
		switch (message.type) {
			case 'i18nStrings':
				// 接收国际化字符串
				i18nStrings = message.strings || {};
				break;
		}
	});

	try {
		// 动态获取 Mermaid 模块 URI
		const scriptTag = document.querySelector('script[data-mermaid-uri]');
		const mermaidUri = scriptTag ? scriptTag.getAttribute('data-mermaid-uri') : null;

		if (!mermaidUri) {
			console.error('Mermaid URI not found');
			return;
		}

		const mermaidModule = await import(mermaidUri);
		const mermaid = mermaidModule.default || mermaidModule;

		console.log('Mermaid loaded:', mermaid);

		// 配置 Mermaid
		const documentStyle = getComputedStyle(document.documentElement);
		const primaryTextColor = documentStyle.getPropertyValue('--vscode-foreground');
		const primaryBackgroundColor = documentStyle.getPropertyValue('--vscode-editor-background');
		const secondaryBackgroundColor = documentStyle.getPropertyValue('--vscode-editorWidget-background');
		const primaryBorderColor = documentStyle.getPropertyValue('--vscode-panel-border');
		const secondaryTextColor = documentStyle.getPropertyValue('--vscode-descriptionForeground');

		await mermaid.initialize({
			startOnLoad: false, // 手动控制渲染
			theme: 'base',
			themeVariables: {
				primaryColor: primaryBackgroundColor,
				primaryTextColor: primaryTextColor,
				primaryBorderColor: primaryBorderColor,
				lineColor: secondaryTextColor,
				secondaryColor: secondaryBackgroundColor,
				tertiaryColor: secondaryTextColor,
			},
			securityLevel: 'loose',
			fontFamily: 'var(--vscode-font-family, sans-serif)',
			flowchart: {
				useMaxWidth: true,
				htmlLabels: true
			},
			sequence: {
				useMaxWidth: true
			},
			gantt: {
				useMaxWidth: true
			},
			er: {
				useMaxWidth: true
			},
			journey: {
				useMaxWidth: true
			}
		});

		// 移除 mermaid 图表中的样式定义
		function removeStyleDefinitions(graphDefinition) {
			// 移除所有 style 行
			return graphDefinition.replace(/^\s*style\s+\w+\s+fill:#[^,]+,stroke:#[^\n]*$/gm, '').trim();
		}

		// 渲染所有 mermaid 图表
		function renderMermaidDiagrams() {
			const mermaidElements = document.querySelectorAll('.mermaid, code.language-mermaid, pre code.language-mermaid');
			console.log('Found mermaid elements:', mermaidElements.length);

			// 统计数据
			let chartTotalCount = mermaidElements.length;
			let chartInvalidCount = 0;

			const renderPromises = Array.from(mermaidElements).map(async (element, index) => {
				const graphId = 'mermaid-graph-' + index + '-' + Date.now();
				try {
					let graphDefinition = '';

					// 处理不同的元素类型
					if (element.tagName === 'CODE') {
						graphDefinition = element.textContent || element.innerText;
						// 将 code 元素替换为 div
						const div = document.createElement('div');
						div.className = 'mermaid';
						div.textContent = graphDefinition;
						element.parentNode.replaceChild(div, element);
						element = div;
					} else {
						graphDefinition = element.textContent || element.innerText;
					}

					if (graphDefinition.trim()) {
						// 移除样式定义
						graphDefinition = removeStyleDefinitions(graphDefinition);
						console.log('Rendering mermaid graph:', graphId, graphDefinition.substring(0, 50) + '...');

						const { svg } = await mermaid.render(graphId, graphDefinition);
						element.innerHTML = svg;
						element.removeAttribute('data-processed');
						console.log('Successfully rendered mermaid graph:', graphId);
					}
				} catch (error) {
					console.error('Error rendering mermaid diagram:', error);
					chartInvalidCount++;
					const errorDiv = document.querySelector(`#d${graphId}`);
					errorDiv && errorDiv.remove();
					element.innerHTML = `
						<div style="
						padding: 20px;
						text-align: center;
						color: var(--vscode-editor-foreground);
						background-color: var(--vscode-textCodeBlock-background);
						font-size: 14px;
						">
						<div style="margin-bottom: 8px;">📊 ${t('Mermaid Chart')}</div>
						<div style="font-size: 12px;">${t('Chart render error, please check the syntax')}</div>
						</div>
					`;
				}
			});

			// 等待所有渲染完成后发送埋点数据
			Promise.all(renderPromises).then(() => {
				// 发送埋点数据到后端
				if (chartTotalCount > 0 && getVsCodeApi()) {
					getVsCodeApi().postMessage({
						type: 'mermaidTrack',
						eventType: 'repowiki-mermaid-chart-render',
						eventData: {
							chart_total_count: chartTotalCount,
							chart_invalid_count: chartInvalidCount
						}
					});
				}
			});
		}

		// DOM 加载完成后渲染
		if (document.readyState === 'loading') {
			document.addEventListener('DOMContentLoaded', renderMermaidDiagrams);
		} else {
			renderMermaidDiagrams();
		}

		// 主题变化监听
		const observer = new MutationObserver(function (mutations) {
			mutations.forEach(function (mutation) {
				if (mutation.type === 'attributes' && mutation.attributeName === 'class') {
					const documentStyle = getComputedStyle(document.documentElement);
					const primaryTextColor = documentStyle.getPropertyValue('--vscode-foreground');
					const primaryBackgroundColor = documentStyle.getPropertyValue('--vscode-editor-background');
					const secondaryBackgroundColor = documentStyle.getPropertyValue('--vscode-editorWidget-background');
					const primaryBorderColor = documentStyle.getPropertyValue('--vscode-panel-border');
					const secondaryTextColor = documentStyle.getPropertyValue('--vscode-descriptionForeground');

					mermaid.initialize({
						startOnLoad: false, // 手动控制渲染
						theme: 'base',
						themeVariables: {
							primaryColor: primaryBackgroundColor,
							primaryTextColor: primaryTextColor,
							primaryBorderColor: primaryBorderColor,
							lineColor: secondaryTextColor,
							secondaryColor: secondaryBackgroundColor,
							tertiaryColor: secondaryTextColor,
						},
						securityLevel: 'loose',
						fontFamily: 'var(--vscode-font-family, sans-serif)',
						flowchart: {
							useMaxWidth: true,
							htmlLabels: true
						},
						sequence: {
							useMaxWidth: true
						},
						gantt: {
							useMaxWidth: true
						},
						er: {
							useMaxWidth: true
						},
						journey: {
							useMaxWidth: true
						}
					}).then(() => {
						renderMermaidDiagrams();
					});
				}
			});
		});

		observer.observe(document.body, {
			attributes: true,
			attributeFilter: ['class']
		});

		// 内容变化监听（如果内容动态更新）
		const contentObserver = new MutationObserver(function (mutations) {
			let shouldRender = false;
			mutations.forEach(function (mutation) {
				if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
					for (let node of mutation.addedNodes) {
						if (node.nodeType === Node.ELEMENT_NODE) {
							if (node.querySelector && (node.querySelector('.mermaid, code.language-mermaid') ||
								node.classList && (node.classList.contains('mermaid') || node.classList.contains('language-mermaid')))) {
								shouldRender = true;
								break;
							}
						}
					}
				}
			});
			if (shouldRender) {
				setTimeout(renderMermaidDiagrams, 100);
			}
		});

		const contentElement = document.getElementById('content');
		if (contentElement) {
			contentObserver.observe(contentElement, {
				childList: true,
				subtree: true
			});
		}

		// 导出全局函数供外部使用
		window.renderMermaidDiagrams = renderMermaidDiagrams;

	} catch (error) {
		console.error('Failed to load Mermaid:', error);
	}
})();
