(function () {
	'use strict';

	// 获取 VS Code API
	const vscode = acquireVsCodeApi();

	// 国际化字符串 - 这些将由扩展主进程传递过来
	let i18nStrings = {};

	// 国际化函数
	function t(key, ...args) {
		let text = i18nStrings[key] || key;
		// 简单的参数替换，支持 {0}, {1} 等占位符
		if (args.length > 0) {
			args.forEach((arg, index) => {
				text = text.replace(new RegExp(`\\{${index}\\}`, 'g'), arg);
			});
		}
		return text;
	}

	// 状态文本映射
	const statusTextMap = {
		'WAITING': () => t('waiting'),
		'LINING': () => t('queuing'),
		'INITIALIZING': () => t('initializing'),
		'DIRECTORY_GENERATING': () => t('generating_directory'),
		'DOCUMENT_GENERATING': () => t('generating_document'),
		'PAUSED': () => t('paused'),
		'COMPLETED': () => t('completed'),
		'PARTLY_COMPLETED': () => t('partly_completed'),
		'INITIALIZING_FAILED': () => t('initialization_failed'),
		'DIRECTORY_FAILED': () => t('directory_generation_failed'),
		'DOCUMENT_FAILED': () => t('document_generation_failed'),
		// 没有初始化源码由 IDE 控制，所以显示等待
		'NOT_SOURCE_CONTROL_FAILED': () => t('not_source_control'),
		'NO_FILE_FAILED': () => t('no_file_failed'),
		'TOO_MANY_FILES_FAILED': () => t('too_many_files'),
		'TOO_MANY_WORKSPACE_FAILED': () => t('Repository Limit Exceeded'),
		'NO_COMMIT_HISTORY_FAILED': () => t('No Commit History')
	};

	// 状态描述映射
	const statusDescriptionMap = {
		'WAITING': () => t('waiting_to_start_analysis'),
		'LINING': () => t('queuing_for_processing'),
		'INITIALIZING': () => t('initializing_analysis_system'),
		'DIRECTORY_GENERATING': () => t('generating_document_directory_structure'),
		'DOCUMENT_GENERATING': () => t('generating_document_content'),
		'PAUSED': () => t('generation_process_paused'),
		'COMPLETED': () => t('document_generation_completed'),
		'PARTLY_COMPLETED': () => t('generating_document_content'),
		'INITIALIZING_FAILED': () => t('error_during_initialization'),
		'DIRECTORY_FAILED': () => t('error_during_directory_generation'),
		'DOCUMENT_FAILED': () => t('error_during_document_generation'),
		'NOT_SOURCE_CONTROL_FAILED': () => t('project_not_in_version_control'),
		'NO_FILE_FAILED': () => t('no_file_failed_description'),
		'TOO_MANY_FILES_FAILED': () => t('project_files_exceed_limit'),
		'TOO_MANY_WORKSPACE_FAILED': () => t('You have reached the maximum number of code repositories allowed for analysis.'),
		'NO_COMMIT_HISTORY_FAILED': () => t('The current project has no commit history, so the Wiki cannot be generated. Please make at least one commit.')
	};

	// 生成进度描述
	function generateProgressDescription(status, statusDetails) {
		if (!statusDetails) {
			return (statusDescriptionMap[status] && statusDescriptionMap[status]()) || t('unknown_status');
		}

		const { total_catalogs, completed_catalogs, failed_catalogs, processing_catalogs, pending_catalogs } = statusDetails;

		switch (status) {
			case 'DIRECTORY_GENERATING':
			case 'DOCUMENT_GENERATING':
				if (total_catalogs > 0) {
					const progress = ((completed_catalogs / total_catalogs) * 100).toFixed(1);
					return t('progress_details', completed_catalogs, total_catalogs, progress, processing_catalogs, failed_catalogs);
				}
				return statusDescriptionMap[status]();
			case 'PARTLY_COMPLETED':
			case 'COMPLETED':
				if (total_catalogs > 0) {
					return t('completion_details', completed_catalogs, total_catalogs, failed_catalogs);
				}
				return statusDescriptionMap[status]();
			default:
				return (statusDescriptionMap[status] && statusDescriptionMap[status]()) || t('unknown_status');
		}
	}

	// 消息发送函数
	function sendMessage(type, value = null) {
		vscode.postMessage({ type, value });
	}

	// 当前状态变量
	let currentStatus = 'WAITING';
	// 按钮loading状态
	let isButtonLoading = false;
	// 上一次的状态，用于检测状态是否真正发生变化
	let lastStatus = null;

	// 获取按钮配置
	function getButtonConfig(status) {
		switch (status) {
			case 'INITIALIZING':
			case 'DIRECTORY_GENERATING':
			case 'DOCUMENT_GENERATING':
				return {
					text: t('pause'),
					action: 'pause',
					className: 'card_operation_button',
					visible: true
				};
			case 'PAUSED':
				return {
					text: t('start'),
					action: 'resume',
					className: 'card_operation_button',
					visible: true
				};
			case 'INITIALIZING_FAILED':
			case 'DIRECTORY_FAILED':
			case 'DOCUMENT_FAILED':
			case 'PARTLY_COMPLETED':
				return {
					text: t('retry'),
					action: 'retry',
					className: 'card_operation_button retry-button',
					visible: true
				};
			default:
				return {
					text: '',
					action: '',
					className: 'card_operation_button',
					visible: false
				};
		}
	}

	// 设置按钮loading状态
	function setButtonLoading(loading) {
		const button = document.getElementById('pauseBtn');
		if (!button) return;

		isButtonLoading = loading;

		if (loading) {
			button.disabled = true;
			button.classList.add('loading');
		} else {
			button.disabled = false;
			button.classList.remove('loading');
		}
	}

	// 更新按钮显示和功能
	function updateButton(status) {
		const button = document.getElementById('pauseBtn');
		if (!button) return;

		const config = getButtonConfig(status);

		if (config.visible) {
			button.style.display = 'block';
			button.textContent = config.text;
			button.className = config.className;

			// 只有状态真正发生变化时才清除loading状态
			if (lastStatus !== null && lastStatus !== status) {
				setButtonLoading(false);
			}

			// 移除所有事件监听器
			const newButton = button.cloneNode(true);
			button.parentNode.replaceChild(newButton, button);

			// 添加新的事件监听器
			newButton.addEventListener('click', () => {
				if (!isButtonLoading) {
					handleButtonClick(config.action);
				}
			});
		} else {
			button.style.display = 'none';
		}
	}

	// 处理按钮点击事件
	function handleButtonClick(action) {
		// 设置按钮为loading状态
		setButtonLoading(true);

		switch (action) {
			case 'pause':
				sendMessage('repoWiki.pause');
				break;
			case 'resume':
				sendMessage('repoWiki.resume');
				break;
			case 'retry':
				sendMessage('repoWiki.retry');
				break;
		}

		// 延迟一小段时间后拉取状态，给后端操作时间
		setTimeout(() => {
			sendMessage('repoWiki.getStatus');
		}, 500);
	}

	// 更新图标显示
	function updateIconDisplay(status) {
		const iconContainer = document.querySelector('.card_progress_icon');
		if (!iconContainer) return;

		// 清除现有内容和样式
		iconContainer.innerHTML = '';
		iconContainer.className = 'card_progress_icon';

		let iconUri = '';
		let shouldRotate = false;

		// 根据状态选择图标
		switch (status) {
			case 'WAITING':
			case 'LINING':
			case 'INITIALIZING':
			case 'DIRECTORY_GENERATING':
			case 'DOCUMENT_GENERATING':
				iconUri = window.iconUris?.loading || '';
				shouldRotate = true;
				break;
			case 'PAUSED':
				iconUri = window.iconUris?.pause || '';
				break;
			case 'TOO_MANY_FILES_FAILED':
			case 'TOO_MANY_WORKSPACE_FAILED':
			case 'NO_FILE_FAILED':
			case 'NO_COMMIT_HISTORY_FAILED':
			case 'NOT_SOURCE_CONTROL_FAILED':
				iconUri = window.iconUris?.warning || '';
				break;
			case 'INITIALIZING_FAILED':
			case 'DIRECTORY_FAILED':
			case 'DOCUMENT_FAILED':
				iconUri = window.iconUris?.failed || '';
				break;
			default:
				// 对于 COMPLETED 、PARTLY_COMPLETED 或其他状态，不显示图标
				iconContainer.classList.remove('show');
				return;
		}

		if (iconUri) {
			const img = document.createElement('img');
			img.src = iconUri;
			img.alt = status;
			iconContainer.appendChild(img);
			iconContainer.classList.add('show');

			if (shouldRotate) {
				img.classList.add('rotating');
			}
		}
	}

	// 更新状态显示
	function updateStatusDisplay(status, statusDetails) {
		const progressText = document.querySelector('.card_progress_text');
		const progressDescription = document.querySelector('.card_progress_description');

		if (progressText) {
			progressText.textContent = (statusTextMap[status] && statusTextMap[status]()) || t('unknown_status');
		}
		if (progressDescription) {
			progressDescription.textContent = generateProgressDescription(status, statusDetails);
		}

		// 更新图标显示
		updateIconDisplay(status);

		// 记录上一次的状态
		lastStatus = currentStatus;
		// 更新当前状态并更新按钮
		currentStatus = status;
		updateButton(status);
	}

	// 监听来自后端的消息
	window.addEventListener('message', event => {
		const message = event.data;
		switch (message.type) {
			case 'statusUpdate':
				updateStatusDisplay(message.status, message.statusDetails);
				break;
			case 'i18nStrings':
				// 接收国际化字符串
				i18nStrings = message.strings || {};
				break;
		}
	});



	// 事件绑定函数
	function bindEventListeners() {
		// 初始化动态按钮和图标
		updateButton(currentStatus);
		updateIconDisplay(currentStatus);
	}

	// 初始化函数
	function initialize() {
		if (document.readyState === 'loading') {
			document.addEventListener('DOMContentLoaded', function () {
				bindEventListeners();
			});
		} else {
			bindEventListeners();
		}
	}

	// 启动应用
	initialize();
})();
