"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.activate = activate;
exports.deactivate = deactivate;
const vscode = __importStar(require("vscode"));
const RepoWikiTreeProvider_1 = require("./providers/RepoWikiTreeProvider");
const RepoWikiMarkdownViewProvider_1 = require("./providers/RepoWikiMarkdownViewProvider");
const RepoWikiWebviewProvider_1 = require("./providers/RepoWikiWebviewProvider");
const l10nUtils_1 = require("./utils/l10nUtils");
const StatusManager_1 = require("./utils/StatusManager");
const l10n = __importStar(require("@vscode/l10n"));
/**
 * 处理HTML中的file://链接，将它们转换为可点击的链接
 */
function processFileLinks(html) {
    // 先处理 markdown 链接格式 [text](file://url)
    let result = html.replace(/\[([^\]]+)\]\(file:\/\/([^)]+)\)/g, (_match, text, filePath) => {
        return `<a href="file://${filePath}">${text}</a>`;
    });
    // 然后处理裸露的 file:// 链接（不在已有的链接标签中）
    result = result.replace(/(^|[^"'])file:\/\/([^)\s<>"']+)/g, (_match, prefix, filePath) => {
        return `${prefix}<a href="file://${filePath}">${filePath}</a>`;
    });
    return result;
}
async function activate(context) {
    // 初始化l10n
    l10nUtils_1.L10nUtils.ensureInitialize(context);
    const qoder = vscode.extensions.getExtension('aicoding.aicoding-agent');
    if (!qoder || !qoder.isActive) {
        return;
    }
    const qoderApi = qoder.exports;
    // 创建统一的状态管理器
    const statusManager = new StatusManager_1.StatusManager(context, qoderApi);
    // 设置初始状态
    vscode.commands.executeCommand('setContext', 'repoWiki.generationInProgress', true);
    // 注册树形视图提供器
    const treeProvider = new RepoWikiTreeProvider_1.RepoWikiTreeProvider(context, qoderApi, statusManager);
    const treeView = vscode.window.createTreeView('repoWiki.fileTree', {
        treeDataProvider: treeProvider,
        showCollapseAll: true
    });
    // 监听树形视图可见性变化
    treeView.onDidChangeVisibility((e) => {
        // 通知 TreeProvider 可见性状态变化
        treeProvider.setVisible(e.visible);
    });
    // 监听树形视图选择变化（当用户点击树形视图时）
    treeView.onDidChangeSelection(() => {
        // 当用户与树形视图交互时，认为它是激活的
        statusManager.setComponentVisibility('tree', true);
    });
    // 监听视图焦点变化
    treeView.onDidCollapseElement(() => {
        // 当用户折叠/展开元素时，认为树形视图是激活的
        statusManager.setComponentVisibility('tree', true);
    });
    treeView.onDidExpandElement(() => {
        // 当用户折叠/展开元素时，认为树形视图是激活的
        statusManager.setComponentVisibility('tree', true);
    });
    // 注册 webview 视图提供器
    const webviewProvider = new RepoWikiWebviewProvider_1.RepoWikiOverviewWebviewProvider(context.extensionUri, qoderApi, statusManager, context);
    context.subscriptions.push(vscode.window.registerWebviewViewProvider(RepoWikiWebviewProvider_1.RepoWikiOverviewWebviewProvider.viewType, webviewProvider, {
        webviewOptions: {
            retainContextWhenHidden: true
        }
    }));
    // 设置组件引用到 StatusManager，用于定期检查实际可见性
    statusManager.setComponentReferences(treeView, webviewProvider);
    // 监听窗口状态变化，用于更精确的可见性检测
    vscode.window.onDidChangeWindowState((windowState) => {
        if (!windowState.focused) {
            // 窗口失去焦点时，停止轮询
            statusManager.setComponentVisibility('tree', false);
            statusManager.setComponentVisibility('webview', false);
        }
        else {
            // 窗口获得焦点时，检查当前活动的视图
            // 延迟一下以确保状态已经更新
            setTimeout(() => {
                // 重新评估可见性
                statusManager.setComponentVisibility('tree', treeView.visible);
                statusManager.setComponentVisibility('webview', webviewProvider.visible);
            }, 100);
        }
    });
    // 注册文件装饰器，用于给禁用的项目设置灰色
    const fileDecorationProvider = vscode.window.registerFileDecorationProvider({
        provideFileDecoration(uri) {
            if (uri.scheme === 'repo-wiki-disabled') {
                return {
                    color: new vscode.ThemeColor('disabledForeground'),
                    tooltip: l10n.t('Document not completed, temporarily unavailable')
                };
            }
            return undefined;
        }
    });
    context.subscriptions.push(fileDecorationProvider);
    // 注册命令
    const disposables = [
        // 打开 markdown 文件命令
        vscode.commands.registerCommand('repoWiki.openFile', async (markdownFile) => {
            if (markdownFile) {
                try {
                    // 如果内容未加载，先异步加载
                    let fileWithContent = markdownFile;
                    if (!markdownFile.loaded || !markdownFile.content) {
                        // 异步加载内容
                        fileWithContent = await treeProvider.loadFileContent(markdownFile);
                    }
                    // 使用VSCode内置的markdown渲染API来创建高质量预览
                    const htmlContent = await vscode.commands.executeCommand('markdown.api.render', fileWithContent.content);
                    // 后处理HTML以修复file://链接
                    const processedHtmlContent = processFileLinks(htmlContent);
                    // 创建带有原生渲染内容的webview
                    RepoWikiMarkdownViewProvider_1.RepoWikiMarkdownViewProvider.createOrShow(context.extensionUri, fileWithContent, processedHtmlContent);
                }
                catch (error) {
                    vscode.window.showErrorMessage(l10n.t('Unable to open document: {0}', markdownFile.name));
                }
            }
            else {
                vscode.window.showWarningMessage(l10n.t('Document information unavailable'));
            }
        }),
        // 刷新文件树命令
        vscode.commands.registerCommand('repoWiki.fileTree.refresh', async () => {
            try {
                await treeProvider.refresh();
            }
            catch (error) {
                vscode.window.showErrorMessage(l10n.t('Failed to refresh document list'));
            }
        }),
        // 重试生成命令
        vscode.commands.registerCommand('repoWiki.retryGeneration', async (markdownFile) => {
            try {
                if (!markdownFile?.id) {
                    vscode.window.showErrorMessage(l10n.t('Unable to retry: Document information incomplete'));
                    return;
                }
                // 显示进度提示
                await vscode.window.withProgress({
                    location: vscode.ProgressLocation.Notification,
                    title: l10n.t('Retrying document generation: {0}', markdownFile.name),
                    cancellable: false
                }, async (progress) => {
                    try {
                        // 调用重试 API
                        await qoderApi.repoWiki.regenerateWikiContent(markdownFile.id);
                        progress.report({ increment: 100, message: l10n.t('Retry request sent') });
                        // 刷新文档列表
                        setTimeout(() => {
                            statusManager.refreshCatalogManually();
                        }, 1000);
                    }
                    catch (error) {
                        vscode.window.showErrorMessage(l10n.t('Retry generation failed: {0}', error instanceof Error ? error.message : l10n.t('Unknown error')));
                    }
                });
            }
            catch (error) {
                vscode.window.showErrorMessage(l10n.t('Error occurred during retry generation'));
            }
        })
    ];
    context.subscriptions.push(...disposables);
}
function deactivate() {
}
//# sourceMappingURL=extension.js.map