"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.RepoWikiMarkdownViewProvider = void 0;
const vscode = __importStar(require("vscode"));
const getNonce_1 = require("../utils/getNonce");
const l10n = __importStar(require("@vscode/l10n"));
const Tracker_1 = require("../utils/Tracker");
class RepoWikiMarkdownViewProvider {
    static createOrShow(extensionUri, markdownFile, renderedHtml) {
        const fileName = markdownFile.name;
        // 检查是否已经打开了同名文件
        const existingPanel = RepoWikiMarkdownViewProvider._openPanels.get(fileName);
        if (existingPanel) {
            // 如果已存在，则激活现有面板
            existingPanel.reveal(vscode.ViewColumn.One);
            return;
        }
        const column = vscode.ViewColumn.One;
        // 创建新的webview面板
        const panel = vscode.window.createWebviewPanel('repoWikiMarkdown', `${fileName}`, column, {
            enableScripts: true,
            localResourceRoots: [extensionUri],
            retainContextWhenHidden: true // 防止 webview 失去焦点时被销毁重新渲染
        });
        // 将新面板添加到跟踪Map中
        RepoWikiMarkdownViewProvider._openPanels.set(fileName, panel);
        panel.webview.html = RepoWikiMarkdownViewProvider._getHtmlForWebview(panel.webview, extensionUri, markdownFile, renderedHtml);
        // 监听来自webview的消息
        panel.webview.onDidReceiveMessage(async (message) => {
            switch (message.type) {
                case 'openLink':
                    await RepoWikiMarkdownViewProvider._handleOpenLink(message.href);
                    break;
                case 'mermaidTrack':
                    // 处理 Mermaid 图表渲染统计埋点
                    await (0, Tracker_1.trackToExtension)(message.eventType, message.eventData);
                    break;
            }
        }, undefined, []);
        // 当面板被关闭时，从Map中移除
        panel.onDidDispose(() => {
            RepoWikiMarkdownViewProvider._openPanels.delete(fileName);
        });
    }
    // 处理打开链接的逻辑
    static async _handleOpenLink(href) {
        try {
            // 处理file://链接
            if (href.startsWith('file://')) {
                // 解析链接中的行号信息
                const { filePath, lineRange } = RepoWikiMarkdownViewProvider._parseFileLink(href);
                let uri = vscode.Uri.parse(filePath);
                // 检查是否是相对路径，如果是则补充完整项目地址
                let relativePath = filePath.replace('file://', '');
                // 去掉末尾的斜杠，避免路径末尾出现斜杠
                relativePath = relativePath.replace(/\/$/, '');
                const isAbsolutePath = relativePath.startsWith('/') || /^[a-zA-Z]:/.test(relativePath);
                if (uri.scheme === 'file' && !isAbsolutePath) {
                    const workspaceFolder = vscode.workspace.workspaceFolders?.[0];
                    if (workspaceFolder) {
                        // 相对路径，需要加上项目目录
                        uri = vscode.Uri.joinPath(workspaceFolder.uri, relativePath);
                    }
                }
                // 准备打开选项
                const options = {
                    viewColumn: vscode.ViewColumn.One
                };
                // 如果有行号信息，设置选择范围
                if (lineRange) {
                    const startLine = lineRange.start - 1; // VS Code 行号从0开始
                    const endLine = lineRange.end ? lineRange.end - 1 : startLine;
                    options.selection = new vscode.Range(new vscode.Position(startLine, 0), new vscode.Position(endLine, Number.MAX_SAFE_INTEGER));
                }
                // 尝试打开文件
                await vscode.commands.executeCommand('vscode.open', uri, options);
            }
            else if (href.startsWith('http://') || href.startsWith('https://')) {
                // 处理外部链接
                await vscode.env.openExternal(vscode.Uri.parse(href));
            }
            else {
                // 处理相对路径链接
                const workspaceFolder = vscode.workspace.workspaceFolders?.[0];
                if (workspaceFolder) {
                    // 解析可能的行号信息
                    const { filePath, lineRange } = RepoWikiMarkdownViewProvider._parseFileLink(href);
                    const fullPath = vscode.Uri.joinPath(workspaceFolder.uri, filePath);
                    const options = {
                        viewColumn: vscode.ViewColumn.One
                    };
                    // 如果有行号信息，设置选择范围
                    if (lineRange) {
                        const startLine = lineRange.start - 1; // VS Code 行号从0开始
                        const endLine = lineRange.end ? lineRange.end - 1 : startLine;
                        options.selection = new vscode.Range(new vscode.Position(startLine, 0), new vscode.Position(endLine, Number.MAX_SAFE_INTEGER));
                    }
                    await vscode.commands.executeCommand('vscode.open', fullPath, options);
                }
            }
        }
        catch (error) {
            vscode.window.showErrorMessage(l10n.t('Unable to open link: {0}', href));
        }
    }
    // 解析文件链接中的路径和行号信息
    static _parseFileLink(href) {
        // 检查是否有行号锚点 (#L1, #L1-L20, #L10-L30)
        const hashIndex = href.indexOf('#');
        if (hashIndex === -1) {
            return { filePath: href };
        }
        const filePath = href.substring(0, hashIndex);
        const anchor = href.substring(hashIndex + 1);
        // 匹配行号格式: L1, L1-L20
        const lineMatch = anchor.match(/^L(\d+)(?:-L(\d+))?$/);
        if (lineMatch) {
            const startLine = parseInt(lineMatch[1], 10);
            const endLine = lineMatch[2] ? parseInt(lineMatch[2], 10) : undefined;
            return {
                filePath,
                lineRange: {
                    start: startLine,
                    end: endLine
                }
            };
        }
        // 如果不是行号格式，返回原始路径
        return { filePath };
    }
    static _getHtmlForWebview(webview, extensionUri, markdownFile, renderedHtml) {
        const nonce = (0, getNonce_1.getNonce)();
        // 使用VSCode预渲染的HTML，如果没有则显示错误信息
        const htmlContent = renderedHtml || `<p>${l10n.t('Unable to render markdown content')}</p>`;
        // 获取CSS和JS文件的webview URI
        const markdownCssUri = webview.asWebviewUri(vscode.Uri.joinPath(extensionUri, 'media', 'markdown.css'));
        const highlightCssUri = webview.asWebviewUri(vscode.Uri.joinPath(extensionUri, 'media', 'highlight.css'));
        const tocCssUri = webview.asWebviewUri(vscode.Uri.joinPath(extensionUri, 'media', 'toc.css'));
        const tocJsUri = webview.asWebviewUri(vscode.Uri.joinPath(extensionUri, 'media', 'toc.js'));
        const webviewCommonJsUri = webview.asWebviewUri(vscode.Uri.joinPath(extensionUri, 'media', 'webview-common.js'));
        // 获取 Mermaid 相关文件的 webview URI
        const mermaidJsUri = webview.asWebviewUri(vscode.Uri.joinPath(extensionUri, 'media', 'mermaid-11.0.0', 'mermaid.esm.min.mjs'));
        const mermaidInitJsUri = webview.asWebviewUri(vscode.Uri.joinPath(extensionUri, 'media', 'mermaid-init.js'));
        return `<!DOCTYPE html>
            <html lang="en">
            <head>
                <meta charset="UTF-8">
                <meta http-equiv="Content-Security-Policy" content="default-src 'none'; style-src ${webview.cspSource} 'unsafe-inline'; img-src ${webview.cspSource} https: data: blob:; media-src ${webview.cspSource} https: data: blob:; script-src ${webview.cspSource} 'nonce-${nonce}' 'unsafe-eval'; connect-src ${webview.cspSource};">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title>${markdownFile.name}</title>
                <link rel="stylesheet" href="${markdownCssUri}">
                <link rel="stylesheet" href="${highlightCssUri}">
                <link rel="stylesheet" href="${tocCssUri}">
                <style>
                    .mermaid {
                        display: flex;
                        justify-content: center;
                        margin: 1em 0;
                    }
                    .mermaid svg {
                        max-width: 100%;
                        height: auto;
                    }
                    /* 增强链接样式 */
                    a[href^="file://"] {
                        color: var(--vscode-textLink-foreground);
                        text-decoration: none;
                        cursor: pointer;
                    }
                    a[href^="file://"]:hover {
                        text-decoration: underline;
                        color: var(--vscode-textLink-activeForeground);
                    }
                </style>
            </head>
            <body class="vscode-${vscode.window.activeColorTheme.kind === vscode.ColorThemeKind.Light ? 'light' : 'dark'}">
                <div class="markdown-container">
                    <button class="toc-toggle" id="tocToggle">
                        <svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" fill="currentColor" version="1.1" width="20" height="20" viewBox="0 0 20 20"><defs><clipPath id="master_svg0_259_3309"><rect x="0" y="0" width="20" height="20" rx="0"/></clipPath></defs><g clip-path="url(#master_svg0_259_3309)"><g><path d="M4.62009765625,3.70001220703125L3.10009765625,3.70001220703125L3.10009765625,5.30001220703125L4.62009765625,5.30001220703125L4.62009765625,9.20001220703125L4.60009765625,9.20001220703125L4.60009765625,10.80001220703125L4.62009765625,10.80001220703125L4.62009765625,15.40001220703125C4.62009765625,15.94121220703125,5.05889765625,16.38001220703125,5.60009765625,16.38001220703125L7.75009765625,16.38001220703125L7.75009765625,14.82001220703125L6.18009765625,14.82001220703125L6.18009765625,10.80001220703125L7.70009765625,10.80001220703125L7.70009765625,9.20001220703125L6.18009765625,9.20001220703125L6.18009765625,5.30001220703125L7.70009765625,5.30001220703125L7.70009765625,3.70001220703125L4.62009765625,3.70001220703125ZM16.90009765625,3.70001220703125L9.30009765625,3.70001220703125L9.30009765625,5.30001220703125L16.90009765625,5.30001220703125L16.90009765625,3.70001220703125ZM16.90009765625,9.20001220703125L9.30009765625,9.20001220703125L9.30009765625,10.80001220703125L16.90009765625,10.80001220703125L16.90009765625,9.20001220703125ZM9.30009765625,14.70001220703125L16.90009765625,14.70001220703125L16.90009765625,16.30001220703125L9.30009765625,16.30001220703125L9.30009765625,14.70001220703125Z" fill="currentColor" fill-opacity="0.8799999952316284"/></g></g></svg>
                    </button>

                    <div class="content" id="content">
                        ${htmlContent}
                    </div>

                    <div class="toc-panel" id="tocPanel">
                        <div class="toc-header">
                            <span class="toc-title">${l10n.t('Table of Contents')}</span>
                            <button class="toc-close" id="tocClose">×</button>
                        </div>
                        <div class="toc-content">
                            <ul class="toc-list" id="tocList"></ul>
                            <div class="toc-empty" id="tocEmpty" style="display: none;">
                                ${l10n.t('No headings available')}
                            </div>
                        </div>
                    </div>
                </div>

                <script nonce="${nonce}" src="${webviewCommonJsUri}"></script>
                <script nonce="${nonce}" src="${tocJsUri}"></script>
                <script nonce="${nonce}" data-mermaid-uri="${mermaidJsUri}" src="${mermaidInitJsUri}"></script>
            </body>
            </html>`;
    }
}
exports.RepoWikiMarkdownViewProvider = RepoWikiMarkdownViewProvider;
// 跟踪已打开的面板，键为文件名，值为面板实例
RepoWikiMarkdownViewProvider._openPanels = new Map();
//# sourceMappingURL=RepoWikiMarkdownViewProvider.js.map