"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.RepoWikiTreeProvider = void 0;
const vscode = __importStar(require("vscode"));
const l10n = __importStar(require("@vscode/l10n"));
class RepoWikiTreeProvider {
    constructor(context, qoderApi, statusManager) {
        this.context = context;
        this.qoderApi = qoderApi;
        this.statusManager = statusManager;
        this._onDidChangeTreeData = new vscode.EventEmitter();
        this.onDidChangeTreeData = this._onDidChangeTreeData.event;
        this.wikiFiles = [];
        this.isLoading = false;
        this.isVisible = false; // 记录TreeView是否可见
        // 注册为目录监听器
        this.statusManager.addCatalogListener(this);
        if (this.qoderApi.auth.isLogin()) {
            // 启动时立即加载一次数据
            this.loadWikiCatalogs().then(() => {
                this._onDidChangeTreeData.fire();
            }).catch(error => {
                console.error(l10n.t('Failed to initialize wiki catalog loading'), error);
            });
        }
        // 自动注册清理逻辑
        this.context.subscriptions.push({
            dispose: () => {
                this.statusManager.removeCatalogListener(this);
            }
        });
    }
    /**
 * 从 WikiCatalog 转换为 MarkdownFile
 */
    convertWikiCatalogToMarkdownFile(catalog) {
        const status = catalog.status || 'unknown';
        return {
            id: catalog.id,
            name: catalog.name,
            path: catalog.id, // 使用 id 作为路径
            status: status,
            description: undefined, // 不显示任何描述
            loaded: false,
            children: catalog.subCatalog?.map(subCatalog => this.convertWikiCatalogToMarkdownFile(subCatalog))
        };
    }
    /**
     * 加载文档目录
     */
    async loadWikiCatalogs() {
        if (this.isLoading) {
            return this.wikiFiles;
        }
        this.isLoading = true;
        try {
            const response = await this.qoderApi.repoWiki.getWikiCatalogs();
            if (response.Result) {
                this.wikiFiles = response.Result.map(catalog => this.convertWikiCatalogToMarkdownFile(catalog));
            }
            else {
                this.wikiFiles = [];
            }
        }
        catch (error) {
            console.error(l10n.t('Failed to load wiki catalogs'), error);
            this.wikiFiles = [];
        }
        finally {
            this.isLoading = false;
        }
        return this.wikiFiles;
    }
    /**
     * 异步加载文档内容
     * @param file 要加载内容的文档
     * @returns 返回包含内容的文档对象
     */
    async loadFileContent(file) {
        // 如果已经加载过了，直接返回
        if (file.loaded && file.content) {
            return file;
        }
        // 正在加载中，等待加载完成
        if (file.loading) {
            return new Promise((resolve) => {
                const checkLoading = setInterval(() => {
                    if (!file.loading && file.loaded) {
                        clearInterval(checkLoading);
                        resolve(file);
                    }
                }, 100);
            });
        }
        // 设置加载状态
        file.loading = true;
        try {
            // 调用获取文档内容的 API
            const response = await this.qoderApi.repoWiki.getWikiContent(file.id);
            if (response.Result?.content) {
                // 使用 API 返回的实际内容
                file.content = response.Result.content;
            }
            else {
                // 如果 API 返回为空，使用占位内容
                file.content = `# ${file.name}

${l10n.t('Document ID')}: ${file.id}
${l10n.t('Status')}: ${file.status}

*${l10n.t('No content available')}*`;
            }
            // 更新文档对象
            file.loaded = true;
            file.loading = false;
            return file;
        }
        catch (error) {
            console.error(l10n.t('Failed to load wiki content'), error);
            // 错误时显示错误占位内容
            file.content = `# ${file.name}

${l10n.t('Document ID')}: ${file.id}
${l10n.t('Status')}: ${file.status}

*${l10n.t('Load failed: {0}', error instanceof Error ? error.message : l10n.t('Unknown error'))}*`;
            file.loaded = true;
            file.loading = false;
            // 显示错误提示
            vscode.window.showErrorMessage(l10n.t('Failed to load document content: {0}', file.name));
            return file;
        }
    }
    /**
     * 实现 CatalogListener 接口
     * 当目录需要更新时被调用
     */
    onCatalogUpdate() {
        // 异步刷新目录
        this.autoRefreshWikiCatalogs().catch(error => {
            console.error('目录更新失败', error);
        });
    }
    /**
     * 设置TreeView可见性状态
     * @param visible 是否可见
     */
    setVisible(visible) {
        if (this.isVisible === visible) {
            return; // 状态没有变化，直接返回
        }
        this.isVisible = visible;
        // 通知状态管理器组件可见性变化
        this.statusManager.setComponentVisibility('tree', visible);
        if (visible) {
            // 立即刷新一次数据
            this.refresh();
        }
    }
    /**
     * 自动刷新文档目录（定时调用）
     */
    async autoRefreshWikiCatalogs() {
        try {
            // 只有在可见且不是手动刷新状态下才进行自动刷新
            if (!this.isVisible || this.isLoading) {
                return;
            }
            await this.loadWikiCatalogs();
            // 触发 UI 更新
            this._onDidChangeTreeData.fire();
        }
        catch (error) {
            console.error(l10n.t('Auto refresh wiki catalog failed'), error);
            // 自动刷新失败时不显示错误信息，避免干扰用户
        }
    }
    /**
     * 处理来自webview的状态请求
     */
    async handleStatusRequest() {
        // 直接调用 StatusManager 的立即获取状态方法
        await this.statusManager.fetchStatusImmediately();
    }
    /**
     * 手动刷新文档目录
     */
    async refresh() {
        try {
            // 清空缓存，重置状态
            this.wikiFiles = [];
            this.isLoading = false;
            // 主动加载文档目录
            await this.loadWikiCatalogs();
            // 触发 UI 更新
            this._onDidChangeTreeData.fire();
        }
        catch (error) {
            console.error(l10n.t('Manual refresh wiki catalog failed'), error);
            vscode.window.showErrorMessage(l10n.t('Failed to refresh wiki catalog'));
        }
    }
    getTreeItem(element) {
        // 所有节点都是文档，有子文档的可展开，没有子文档的不可展开
        const hasChildren = element.children && element.children.length > 0;
        const item = new vscode.TreeItem(element.name, hasChildren
            ? vscode.TreeItemCollapsibleState.Expanded // 有子文档自动展开
            : vscode.TreeItemCollapsibleState.None // 无子文档不可展开
        );
        item.description = element.description;
        // 不显示图标
        item.iconPath = undefined;
        // 只有 completed 状态的文档可以点击打开
        if (element.status === 'completed') {
            item.command = {
                command: 'repoWiki.openFile',
                title: l10n.t('Open Markdown File'),
                arguments: [element]
            };
        }
        else if (element.status === 'failed') {
            // failed 状态的文档设置 contextValue 以显示重试按钮
            item.contextValue = 'failedWikiItem';
            // 不可点击的文档使用自定义的 scheme 来触发文件装饰器
            item.resourceUri = vscode.Uri.parse(`repo-wiki-disabled:${element.name}`);
        }
        else {
            // 其他状态不设置 contextValue，不显示任何按钮
            // 不可点击的文档使用自定义的 scheme 来触发文件装饰器
            item.resourceUri = vscode.Uri.parse(`repo-wiki-disabled:${element.name}`);
        }
        return item;
    }
    async getChildren(element) {
        if (!element) {
            // 根节点，返回顶级文档列表
            return await this.loadWikiCatalogs();
        }
        // 返回子文档
        return element.children || [];
    }
    getParent() {
        return null;
    }
}
exports.RepoWikiTreeProvider = RepoWikiTreeProvider;
//# sourceMappingURL=RepoWikiTreeProvider.js.map