"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.StatusManager = void 0;
const vscode = __importStar(require("vscode"));
const Tracker_1 = require("./Tracker");
/**
 * 统一的状态管理器
 * 负责管理 RepoWiki 的状态轮询和目录刷新
 */
class StatusManager {
    constructor(context, qoderApi) {
        this.context = context;
        this.qoderApi = qoderApi;
        this.isPolling = false;
        this.statusListeners = new Set();
        this.catalogListeners = new Set();
        this.hasAnyVisibleComponent = false;
        this.isLoadingCatalog = false;
        // 直接跟踪组件的可见性状态
        this.treeVisible = false;
        this.webviewVisible = false;
        // 自动注册清理逻辑
        this.context.subscriptions.push({
            dispose: () => {
                this.stopPolling();
                this.stopVisibilityCheck();
            }
        });
        // 启动定期可见性检查
        this.startVisibilityCheck();
    }
    /**
     * 注册状态监听器
     */
    addStatusListener(listener) {
        this.statusListeners.add(listener);
    }
    /**
     * 移除状态监听器
     */
    removeStatusListener(listener) {
        this.statusListeners.delete(listener);
    }
    /**
     * 注册目录监听器
     */
    addCatalogListener(listener) {
        this.catalogListeners.add(listener);
    }
    /**
     * 移除目录监听器
     */
    removeCatalogListener(listener) {
        this.catalogListeners.delete(listener);
    }
    /**
     * 设置外部组件引用，用于检查实际可见性
     */
    setComponentReferences(treeView, webviewProvider) {
        this.treeViewRef = treeView;
        this.webviewProviderRef = webviewProvider;
    }
    /**
     * 启动定期可见性检查
     */
    startVisibilityCheck() {
        // 每3秒检查一次实际的可见性状态
        this.visibilityCheckTimer = setInterval(() => {
            this.checkActualVisibility();
        }, 3000);
    }
    /**
     * 停止定期可见性检查
     */
    stopVisibilityCheck() {
        if (this.visibilityCheckTimer) {
            clearInterval(this.visibilityCheckTimer);
            this.visibilityCheckTimer = undefined;
        }
    }
    /**
     * 检查组件的实际可见性状态
     */
    checkActualVisibility() {
        let actualTreeVisible = false;
        let actualWebviewVisible = false;
        // 检查 TreeView 的实际可见性
        if (this.treeViewRef) {
            actualTreeVisible = this.treeViewRef.visible;
        }
        // 检查 Webview 的实际可见性
        if (this.webviewProviderRef && this.webviewProviderRef.visible !== undefined) {
            actualWebviewVisible = this.webviewProviderRef.visible;
        }
        // 如果检测到状态变化，更新状态
        if (actualTreeVisible !== this.treeVisible) {
            this.setComponentVisibility('tree', actualTreeVisible);
        }
        if (actualWebviewVisible !== this.webviewVisible) {
            this.setComponentVisibility('webview', actualWebviewVisible);
        }
    }
    /**
     * 设置组件可见性
     * @param componentName 组件名称（tree 或 webview）
     * @param visible 是否可见
     */
    setComponentVisibility(componentName, visible) {
        // 检查状态是否真的发生了变化
        const currentComponentVisible = componentName === 'tree' ? this.treeVisible : this.webviewVisible;
        if (currentComponentVisible === visible) {
            return; // 状态没有变化，直接返回
        }
        // 更新对应组件的可见性状态
        if (componentName === 'tree') {
            this.treeVisible = visible;
        }
        else {
            this.webviewVisible = visible;
        }
        const previouslyVisible = this.hasAnyVisibleComponent;
        this.hasAnyVisibleComponent = this.treeVisible || this.webviewVisible;
        // 状态变化时更新轮询
        if (this.hasAnyVisibleComponent !== previouslyVisible) {
            if (this.hasAnyVisibleComponent) {
                this.startPolling();
                // 立即获取一次状态和目录
                this.fetchStatusImmediately();
                this.notifyCatalogUpdate();
                // 如果是 tree 变为可见，发送曝光埋点
                if (componentName === 'tree' && visible) {
                    this.trackTreeViewExposure();
                }
            }
            else {
                this.stopPolling();
            }
        }
    }
    /**
     * 获取 TreeView 是否可见
     */
    get isTreeVisible() {
        return this.treeVisible;
    }
    /**
     * 获取 Webview 是否可见
     */
    get isWebviewVisible() {
        return this.webviewVisible;
    }
    /**
     * 启动轮询
     */
    startPolling() {
        if (this.isPolling) {
            return;
        }
        this.isPolling = true;
        // 状态轮询 - 每10秒
        this.statusTimer = setInterval(async () => {
            if (this.hasAnyVisibleComponent) {
                await this.fetchWikiStatus();
            }
        }, 10000);
        // 目录轮询 - 每10秒
        this.catalogTimer = setInterval(async () => {
            if (this.hasAnyVisibleComponent) {
                await this.refreshCatalog();
            }
        }, 10000);
    }
    /**
     * 停止轮询
     */
    stopPolling() {
        if (!this.isPolling) {
            return;
        }
        this.isPolling = false;
        if (this.statusTimer) {
            clearInterval(this.statusTimer);
            this.statusTimer = undefined;
        }
        if (this.catalogTimer) {
            clearInterval(this.catalogTimer);
            this.catalogTimer = undefined;
        }
    }
    /**
     * 获取wiki状态
     */
    async fetchWikiStatus() {
        try {
            if (this.statusListeners.size === 0) {
                return;
            }
            const response = await this.qoderApi.repoWiki.getWikiStatus();
            let status;
            let statusDetails = undefined;
            // 检查响应是否成功
            if (response.errorCode !== '0' && response.errorCode !== '') {
                status = 'INITIALIZING_FAILED';
            }
            else {
                status = response.Result.wiki_present_status;
                statusDetails = response.Result.status_details;
            }
            // 更新 VS Code context
            await this.updateVSCodeContext(status);
            // 通知所有状态监听器
            this.statusListeners.forEach(listener => {
                try {
                    listener.onStatusUpdate(status, statusDetails);
                }
                catch (error) {
                    console.error('StatusManager: 通知状态监听器失败', error);
                }
            });
        }
        catch (error) {
            // 出错时通知所有监听器
            await this.updateVSCodeContext('INITIALIZING_FAILED');
            this.statusListeners.forEach(listener => {
                try {
                    listener.onStatusUpdate('INITIALIZING_FAILED');
                }
                catch (listenerError) {
                    console.error('StatusManager: 通知状态监听器失败', listenerError);
                }
            });
        }
    }
    /**
     * 立即获取状态（不检查可见性）
     */
    async fetchStatusImmediately() {
        await this.fetchWikiStatus();
    }
    /**
     * 刷新目录
     */
    async refreshCatalog() {
        if (this.isLoadingCatalog || this.catalogListeners.size === 0) {
            return;
        }
        this.isLoadingCatalog = true;
        try {
            // 通知所有目录监听器
            this.notifyCatalogUpdate();
        }
        finally {
            this.isLoadingCatalog = false;
        }
    }
    /**
     * 通知目录更新
     */
    notifyCatalogUpdate() {
        this.catalogListeners.forEach(listener => {
            try {
                listener.onCatalogUpdate();
            }
            catch (error) {
                console.error('StatusManager: 通知目录监听器失败', error);
            }
        });
    }
    /**
     * 手动刷新目录
     */
    async refreshCatalogManually() {
        this.notifyCatalogUpdate();
    }
    /**
     * 更新 VS Code context
     */
    async updateVSCodeContext(status) {
        try {
            // 当状态不是 COMPLETED 时显示 webview
            const shouldShowWebview = status !== 'COMPLETED';
            await vscode.commands.executeCommand('setContext', 'repoWiki.generationInProgress', shouldShowWebview);
            // 加个延迟，等待 webview 更新后再更新后续状态
            await new Promise(resolve => setTimeout(resolve, 30));
        }
        catch (error) {
            // 出错时默认隐藏 webview，显示树形视图
            await vscode.commands.executeCommand('setContext', 'repoWiki.generationInProgress', false);
        }
    }
    /**
     * 发送TreeView曝光埋点
     */
    async trackTreeViewExposure() {
        try {
            await (0, Tracker_1.trackToExtension)('repowiki-exposure');
        }
        catch (error) {
            console.error('StatusManager: TreeView曝光埋点发送失败', error);
        }
    }
    /**
     * 获取当前轮询状态
     */
    get pollingActive() {
        return this.isPolling;
    }
    /**
     * 获取当前可见状态
     */
    get anyComponentVisible() {
        return this.hasAnyVisibleComponent;
    }
}
exports.StatusManager = StatusManager;
//# sourceMappingURL=StatusManager.js.map