"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.isFirstLaunch = isFirstLaunch;
exports.importUserSettingsFromVSCode = importUserSettingsFromVSCode;
exports.importUserSettingsFromCursor = importUserSettingsFromCursor;
const fs = __importStar(require("fs"));
const os = __importStar(require("os"));
const path = __importStar(require("path"));
const vscode = __importStar(require("vscode"));
const JSON5 = __importStar(require("json5"));
const viewTypes_1 = require("./product/viewTypes");
const i18n_1 = require("./util/i18n");
/**
 * 解析带注释的 JSON 文件
 * 使用 json5 库，支持注释、尾随逗号等扩展语法
 */
function parseJsonWithComments(jsonString) {
    try {
        // 使用 json5 解析，自动处理注释、尾随逗号等
        return JSON5.parse(jsonString);
    }
    catch (error) {
        console.warn('JSON5解析失败:', error);
        throw error;
    }
}
/**
 * 安全的 JSON 解析函数
 * 对于配置文件，优先使用 parseJsonWithComments 处理注释
 * 对于普通 JSON，使用标准 JSON.parse
 */
function safeJsonParse(jsonString, allowComments = false) {
    if (allowComments) {
        return parseJsonWithComments(jsonString);
    }
    try {
        return JSON.parse(jsonString);
    }
    catch (error) {
        // 如果标准解析失败，尝试注释解析作为备选
        console.warn('标准JSON解析失败，尝试注释解析:', error);
        return parseJsonWithComments(jsonString);
    }
}
const aiIdeDevExtensionsDir = path.join(os.homedir(), viewTypes_1.IDE_DATA_FOLDER_NAME, 'extensions');
function isFirstLaunch(context) {
    const stateExists = context.globalState.get(viewTypes_1.FIRST_LAUNCH_KEY);
    // If state is set and is true, it's not first launch
    const isFirst = !stateExists;
    return isFirst;
}
function getAiIdeSettingsDir() {
    const platform = process.platform;
    if (platform === 'win32') {
        return path.join(process.env.APPDATA || '', viewTypes_1.IMPORT_USER_SETTINGS_DIRNAME, 'User');
    }
    else if (platform === 'darwin') {
        return path.join(os.homedir(), 'Library', 'Application Support', viewTypes_1.IMPORT_USER_SETTINGS_DIRNAME, 'User');
    }
    else {
        return path.join(os.homedir(), '.config', viewTypes_1.IMPORT_USER_SETTINGS_DIRNAME, 'User');
    }
}
async function copyDirectoryRecursiveSync(source, destination, exclusions = []) {
    await fs.promises.mkdir(destination, { recursive: true });
    const items = await fs.promises.readdir(source);
    let copiedCount = 0;
    let skippedCount = 0;
    let errorCount = 0;
    for (const item of items) {
        const sourcePath = path.join(source, item);
        const destinationPath = path.join(destination, item);
        const shouldExclude = exclusions.some((exclusion) => sourcePath.toLowerCase().includes(exclusion.toLowerCase()));
        if (shouldExclude) {
            skippedCount++;
        }
        else {
            try {
                const stats = await fs.promises.lstat(sourcePath);
                if (stats.isDirectory()) {
                    await copyDirectoryRecursiveSync(sourcePath, destinationPath, exclusions);
                }
                else {
                    await fs.promises.copyFile(sourcePath, destinationPath);
                }
                copiedCount++;
            }
            catch (err) {
                console.warn(`❌ 复制失败: ${sourcePath} -> ${destinationPath}`, err);
                errorCount++;
            }
        }
    }
}
function getSettingsDir(source) {
    const platform = process.platform;
    const appName = source === 'vscode' ? 'Code' : 'Cursor';
    let settingsDir;
    if (platform === 'win32') {
        settingsDir = path.join(process.env.APPDATA || '', appName, 'User');
    }
    else if (platform === 'darwin') {
        settingsDir = path.join(os.homedir(), 'Library', 'Application Support', appName, 'User');
    }
    else {
        settingsDir = path.join(os.homedir(), '.config', appName, 'User');
    }
    return settingsDir;
}
function getExtensionsDir(source) {
    const extensionsDir = path.join(os.homedir(), source === 'vscode' ? '.vscode' : '.cursor', 'extensions');
    return extensionsDir;
}
const DIRECTORY_NOT_EXIST = 'DirectoryNotExist';
const DirectoryNotExist = 'Source settings directory does not exist';
/**
 * 通用的设置复制函数
 */
async function copySettingsToAiIdeDir(source, context) {
    const sourceSettingsDir = getSettingsDir(source);
    const aiIdeDevSettingsDir = getAiIdeSettingsDir();
    // 检查源目录是否存在
    try {
        await fs.promises.access(sourceSettingsDir);
    }
    catch (error) {
        // 如果源目录不存在则退出
        return;
    }
    await fs.promises.mkdir(aiIdeDevSettingsDir, { recursive: true });
    await fs.promises.mkdir(aiIdeDevExtensionsDir, { recursive: true });
    const isFirst = !context?.globalState.get(viewTypes_1.FIRST_LAUNCH_KEY);
    const itemsToCopy = isFirst ? [
        'settings.json',
        'keybindings.json',
        'snippets',
        'sync',
        'globalStorage/state.vscdb',
        'globalStorage/state.vscdb.backup',
    ] : [
        'settings.json',
        'keybindings.json',
        'snippets',
    ];
    const settingsJsonExclusions = [
        'app',
        'window.zoomLevel',
        'workbench.colorTheme',
    ];
    const keybindingsExclusions = [
        'composerMode.agent', // cmd + i
        'composer.sendToAgent', // cmd + l
    ];
    // 需要过滤掉的快捷键列表
    const keybindingsKeyExclusions = [
        'cmd+e',
        'cmd+i',
        'cmd+l',
        'ctrl+e',
        'ctrl+i',
        'ctrl+l'
    ];
    for (const item of itemsToCopy) {
        const sourcePath = path.join(sourceSettingsDir, item);
        const destination = path.join(aiIdeDevSettingsDir, item);
        try {
            const exists = await fs.promises
                .access(sourcePath)
                .then(() => true)
                .catch(() => false);
            if (exists) {
                const stats = await fs.promises.lstat(sourcePath);
                if (stats.isDirectory()) {
                    await copyDirectoryRecursiveSync(sourcePath, destination);
                }
                else {
                    if (item === 'settings.json') {
                        const sourceContent = await fs.promises.readFile(sourcePath, 'utf8');
                        const sourceSettings = safeJsonParse(sourceContent, true);
                        for (const exclusion of settingsJsonExclusions) {
                            delete sourceSettings[exclusion];
                        }
                        let targetAppConfig = {};
                        try {
                            const targetContent = await fs.promises.readFile(destination, 'utf8');
                            const targetSettings = safeJsonParse(targetContent, true);
                            // 保留目标文件中的 app 配置
                            if (targetSettings.app) {
                                targetAppConfig = targetSettings.app;
                            }
                        }
                        catch (err) {
                            // 目标文件不存在或解析失败，使用空对象
                        }
                        // 合并配置：源配置 + 目标的 app 配置
                        const mergedSettings = {
                            ...sourceSettings,
                            ...(Object.keys(targetAppConfig).length > 0 ? { app: targetAppConfig } : {})
                        };
                        // 写入合并后的配置
                        await fs.promises.writeFile(destination, JSON.stringify(mergedSettings, null, 2), 'utf8');
                    }
                    else if (item === 'keybindings.json') {
                        const sourceContent = await fs.promises.readFile(sourcePath, 'utf8');
                        let sourceKeybindings = [];
                        try {
                            sourceKeybindings = safeJsonParse(sourceContent, true);
                        }
                        catch (error) {
                            console.warn('源文件 keybindings.json 解析失败:', error);
                            // 如果解析失败，跳过此文件
                            continue;
                        }
                        // 过滤掉指定的 command 和快捷键
                        const filteredKeybindings = sourceKeybindings.filter((binding) => {
                            // 检查是否在命令排除列表中
                            if (keybindingsExclusions.includes(binding.command)) {
                                return false;
                            }
                            // 检查是否在快捷键排除列表中
                            if (binding.key && keybindingsKeyExclusions.includes(binding.key)) {
                                return false;
                            }
                            return true;
                        });
                        let targetKeybindings = [];
                        try {
                            const targetContent = await fs.promises.readFile(destination, 'utf8');
                            targetKeybindings = safeJsonParse(targetContent, true) || [];
                        }
                        catch (err) {
                            // 目标文件不存在或解析失败，使用空数组
                        }
                        // 合并快捷键配置：目标配置 + 过滤后的源配置
                        // 使用 Set 来避免重复的快捷键绑定（基于 key + command 组合）
                        const existingBindings = new Set(targetKeybindings.map((binding) => `${binding.key}:${binding.command}`));
                        const uniqueSourceBindings = filteredKeybindings.filter((binding) => {
                            return !existingBindings.has(`${binding.key}:${binding.command}`);
                        });
                        const mergedKeybindings = [...targetKeybindings, ...uniqueSourceBindings];
                        // 写入合并后的配置
                        await fs.promises.writeFile(destination, JSON.stringify(mergedKeybindings, null, 2), 'utf8');
                    }
                    else {
                        // 其他文件直接复制
                        await fs.promises.copyFile(sourcePath, destination);
                    }
                }
            }
        }
        catch (error) {
            console.error(`[${source.toUpperCase()}] 复制 ${item} 时出错:`, error);
        }
    }
}
/**
 * 通用的扩展复制函数
 */
async function copyExtensionsToAiIdeDir(source) {
    const sourceExtensionsDir = getExtensionsDir(source);
    const sourceExtensionsJsonDir = path.join(sourceExtensionsDir, 'extensions.json');
    const aiIdeDevExtensionsJsonDir = path.join(aiIdeDevExtensionsDir, 'extensions.json');
    // 检查源扩展目录是否存在
    try {
        await fs.promises.access(sourceExtensionsDir);
    }
    catch (error) {
        return; // 如果源目录不存在，直接返回
    }
    await fs.promises.mkdir(aiIdeDevExtensionsDir, { recursive: true });
    // 基础黑名单
    const baseExclusions = [
        'aicoding.aicoding-agent',
        'alibaba-cloud.tongyi-lingma',
        'ms-ceintl.vscode-language-pack-zh-hans',
        'ms-vscode.remote-server',
        'ms-vscode-remote.remote-ssh',
        'ms-vscode-remote.remote-containers',
        'ms-vscode-remote.remote-wsl',
        'ms-vscode.remote-explorer',
        'ms-vscode-remote.remote-ssh-edit',
        'ms-dotnettools.csharp',
        'ms-vscode.cpptools',
        'ms-python.vscode-pylance',
        'ms-vscode-remote.remote-wsl',
        'ms-vscode.remote-explorer',
        'ms-vscode-remote.remote-containers',
        'ms-vscode-remote.remote-ssh',
        'GitHub.copilot',
        'GitHub.copilot-chat',
        'TabNine.tabnine-vscode',
        'Augment.vscode-augment',
        'BaiduComate.comate',
        'Tencent-Cloud.coding-copilot',
        'saoudrizwan.claude-dev',
        'RooVeterinaryInc.roo-cline',
        'anysphere.remote-ssh',
        'anysphere.remote-containers',
        'anysphere.remote-wsl',
        'anysphere.remote-wsl',
        'anysphere.remote-containers',
        'anysphere.cursorpyright',
        'anysphere.cpptools',
        'anysphere.vscode-tizen-csharp',
        'anysphere.csharp',
        'anysphere.remote-ssh',
    ];
    // Cursor 特有的扩展黑名单
    if (source === 'cursor') {
        // 三个 anysphere.remote 相关的放上面了
    }
    // Add platform specific exclusions
    if (process.platform === 'darwin' && process.arch === 'arm64') {
    }
    // Add platform specific exclusions
    if (process.platform === 'darwin' && process.arch === 'x64') {
    }
    // Add Windows specific exclusions
    if (process.platform === 'win32') {
    }
    // Add Linux specific exclusions
    if (process.platform === 'linux') {
    }
    try {
        await copyDirectoryRecursiveSync(sourceExtensionsDir, aiIdeDevExtensionsDir, 
        // extensions.json 也要排除忽略
        [...baseExclusions, path.join(sourceExtensionsDir, 'extensions.json')]);
        // 合并 extensions.json
        await mergeExtensionsJson(sourceExtensionsJsonDir, aiIdeDevExtensionsJsonDir, baseExclusions);
        // 清理 extensions.json 中磁盘上不存在的扩展
        await cleanupMissingExtensions(aiIdeDevExtensionsJsonDir, aiIdeDevExtensionsDir);
    }
    catch (error) {
        throw error; // 重新抛出错误以便上层处理
    }
}
/**
 * 合并 extensions.json 文件
 * @param sourceExtensionsJsonPath 源 extensions.json 文件路径
 * @param targetExtensionsJsonPath 目标 extensions.json 文件路径
 * @param exclusions 需要过滤掉的扩展ID列表
 */
async function mergeExtensionsJson(sourceExtensionsJsonPath, targetExtensionsJsonPath, exclusions) {
    let sourceExtensions = [];
    let targetExtensions = [];
    // 读取源文件
    try {
        const sourceContent = await fs.promises.readFile(sourceExtensionsJsonPath, 'utf8');
        sourceExtensions = safeJsonParse(sourceContent) || [];
    }
    catch (error) {
        // 如果源文件不存在或无法读取，使用空数组
        console.warn(`Failed to read source extensions.json: ${error}`);
    }
    // 读取目标文件
    try {
        const targetContent = await fs.promises.readFile(targetExtensionsJsonPath, 'utf8');
        targetExtensions = safeJsonParse(targetContent) || [];
    }
    catch (error) {
        // 如果目标文件不存在，使用空数组
        console.warn(`Target extensions.json does not exist, will create new one`);
    }
    // 过滤掉排除列表中的扩展（不区分大小写）
    const exclusionsLowerCase = exclusions.map(id => id.toLowerCase());
    const filteredSourceExtensions = sourceExtensions.filter(ext => !exclusionsLowerCase.includes(ext.identifier?.id?.toLowerCase()));
    // 创建目标扩展的ID集合，用于快速查找（不区分大小写）
    const targetExtensionIds = new Set(targetExtensions.map(ext => ext.identifier?.id?.toLowerCase()));
    // 将不重复的源扩展添加到目标扩展中（不区分大小写比较）
    const uniqueSourceExtensions = filteredSourceExtensions.filter(ext => !targetExtensionIds.has(ext.identifier?.id?.toLowerCase()));
    // 合并扩展列表（目标扩展 + 不重复的源扩展）
    const mergedExtensions = [...targetExtensions, ...uniqueSourceExtensions];
    // 确保目标目录存在
    const targetDir = path.dirname(targetExtensionsJsonPath);
    await fs.promises.mkdir(targetDir, { recursive: true });
    // 写入合并后的内容
    await fs.promises.writeFile(targetExtensionsJsonPath, JSON.stringify(mergedExtensions), 'utf8');
}
/**
 * 清理 extensions.json 中磁盘上不存在的扩展
 * 通过检查扩展目录下是否存在 package.json 来判断扩展是否真实存在
 * @param extensionsJsonPath extensions.json 文件路径
 * @param extensionsDir 扩展安装目录
 */
async function cleanupMissingExtensions(extensionsJsonPath, extensionsDir) {
    try {
        // 读取 extensions.json 文件
        const extensionsContent = await fs.promises.readFile(extensionsJsonPath, 'utf8');
        const extensions = safeJsonParse(extensionsContent) || [];
        // 过滤出实际存在的扩展
        const validExtensions = [];
        let removedCount = 0;
        for (const extension of extensions) {
            const relativeLocation = extension.relativeLocation;
            if (!relativeLocation) {
                // 如果扩展ID不存在，跳过该扩展
                continue;
            }
            // 检查扩展目录下是否存在 package.json
            const packageJsonPath = path.join(extensionsDir, relativeLocation, 'package.json');
            try {
                await fs.promises.access(packageJsonPath);
                // 如果 package.json 存在，保留该扩展
                validExtensions.push(extension);
            }
            catch (error) {
                // 如果 package.json 不存在，说明扩展在磁盘上不存在，需要从 extensions.json 中移除
                console.warn(`扩展 ${extension.identifier} 在磁盘上不存在，将从 extensions.json 中移除`);
                removedCount++;
            }
        }
        // 如果有扩展被移除，重新写入 extensions.json
        if (removedCount > 0) {
            await fs.promises.writeFile(extensionsJsonPath, JSON.stringify(validExtensions), 'utf8');
            console.log(`已从 extensions.json 中移除 ${removedCount} 个磁盘上不存在的扩展`);
        }
        else {
            console.log('所有扩展在磁盘上都存在，无需清理');
        }
    }
    catch (error) {
        console.warn(`清理不存在的扩展时出错: ${error}`);
    }
}
/**
 * 通用的用户设置导入函数
 */
async function importUserSettingsFrom(source, context) {
    try {
        await copySettingsToAiIdeDir(source, context);
        await copyExtensionsToAiIdeDir(source);
        return true;
    }
    catch (error) {
        const sourceName = source === 'vscode' ? 'VS Code' : 'Cursor';
        if (error.cause?.key === DIRECTORY_NOT_EXIST) {
            return error.cause;
        }
        vscode.window.showErrorMessage((0, i18n_1.t)(`从 ${sourceName} 导入扩展失败，您可以尝试手动安装您需要的扩展`, `Failed to import extensions from ${sourceName}, you can try to install the extensions you need manually in the marketplace`));
        console.error(`Import${source}Exts-Error: ${error}`);
        return false;
    }
}
/**
 * 从VSCode导入
 */
async function importUserSettingsFromVSCode(context) {
    return await importUserSettingsFrom('vscode', context);
}
/**
 * 从Cursor导入
 */
async function importUserSettingsFromCursor(context) {
    return await importUserSettingsFrom('cursor', context);
}
//# sourceMappingURL=copySettings.js.map