'use strict';

const withDefaults = require('../shared.webpack.config');
const path = require('path');
const webpack = require('webpack');
const CopyWebpackPlugin = require('copy-webpack-plugin');
const { execSync } = require('child_process');


// 固定品牌/环境常量（不依赖 env）
const APPNAME = process.env.APPNAME || 'Qoder';
const APPNAME_CN = process.env.APPNAME_CN || 'Qoder';
const APPNAME_LOWERCASE = APPNAME.toLowerCase();
const COMMAND_PREFIX = 'aicoding';
const EXT_PUBLISHER = 'aicoding';
const EXT_PUBLISHER_LOWERCASE = EXT_PUBLISHER.toLowerCase();
const EXT_NAME = 'aicoding-agent';

// 替换规则（常量）
const REPLACEMENTS = {
	'@@APPNAME@@': APPNAME,
	'@@APPNAME_CN@@': APPNAME_CN,
	'@@APPNAME_LOWERCASE@@': APPNAME_LOWERCASE,
	'@@COMMAND_PREFIX@@': COMMAND_PREFIX,
	'@@EXT_PUBLISHER@@': EXT_PUBLISHER,
	'@@EXT_PUBLISHER_LOWERCASE@@': EXT_PUBLISHER_LOWERCASE,
	'@@EXT_NAME@@': EXT_NAME,
};

const createStringReplaceOptions = () => {
	return Object.entries(REPLACEMENTS).map(([search, replace]) => ({
		search,
		replace,
		flags: 'g'
	}));
};

const applyStringReplacements = (content) => {
	let result = content;
	Object.entries(REPLACEMENTS).forEach(([search, replace]) => {
		const regex = new RegExp(search.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'), 'g');
		result = result.replace(regex, replace);
	});
	return result;
};

// 根据 APPNAME 选择 media 目录
const mediaSourceDir = APPNAME === 'Qoder' ? 'media-qoder' : 'media-lingma';

// 自定义插件：在构建前执行 preTask.js
class PreTaskPlugin {
	constructor(options = {}) {
		this.options = options;
	}

	apply(compiler) {
		compiler.hooks.beforeRun.tapAsync('PreTaskPlugin', (compiler, callback) => {
			console.log('[PreTaskPlugin] Running preTask.js before build...');
			try {
				const preTaskScript = path.resolve(__dirname, 'scripts/preTask.js');
				execSync(`node "${preTaskScript}"`, {
					stdio: 'inherit',
					cwd: __dirname,
					env: { ...process.env, APPNAME }
				});
				console.log('[PreTaskPlugin] preTask.js completed successfully');
				callback();
			} catch (error) {
				console.error('[PreTaskPlugin] Error executing preTask.js:', error);
				callback(error);
			}
		});

		// 也在 watch 模式下的重新编译前执行
		compiler.hooks.watchRun.tapAsync('PreTaskPlugin', (compiler, callback) => {
			console.log('[PreTaskPlugin] Running preTask.js before watch rebuild...');
			try {
				const preTaskScript = path.resolve(__dirname, 'scripts/preTask.js');
				execSync(`node "${preTaskScript}"`, {
					stdio: 'inherit',
					cwd: __dirname,
					env: { ...process.env, APPNAME }
				});
				console.log('[PreTaskPlugin] preTask.js completed successfully');
				callback();
			} catch (error) {
				console.error('[PreTaskPlugin] Error executing preTask.js:', error);
				callback(error);
			}
		});
	}
}

module.exports = {
	context: __dirname,
	target: 'node', // VS Code extensions run in Node.js context
	entry: {
		extension: './src/extension.ts'
	},
	output: {
		path: path.join(__dirname, 'dist'),
		filename: 'extension.js',
		libraryTarget: 'commonjs2', // Required for VS Code extensions
		clean: true, // Clean output directory before build
	},
	externals: {
		vscode: 'commonjs vscode', // Exclude vscode module
	},
	resolve: {
		extensions: ['.ts', '.js', '.node'], // Support TypeScript and JavaScript files
	},
	module: {
		rules: [
			{
				test: /\.ts$/,
				exclude: /node_modules/,
				use: [
					{ loader: 'ts-loader' },
					{
						loader: 'string-replace-loader',
						options: { multiple: createStringReplaceOptions() },
					},
				],
			},
			{
				test: /\.js$/,
				exclude: /node_modules/,
				use: [
					{
						loader: 'string-replace-loader',
						options: { multiple: createStringReplaceOptions() },
					},
				],
			},
			{
				test: /\.node$/,
				loader: 'node-loader',
			},
			{
				test: /\.(png|jpg|jpe?g|gif)$/,
				use: {
					loader: 'file-loader',
					options: {
						outputPath: './image/',
					},
				},
			},
			{
				test: /\.html$/,
				use: [
					{ loader: 'html-loader' },
					{
						loader: 'string-replace-loader',
						options: { multiple: createStringReplaceOptions() },
					},
				],
			},
		],
	},
	plugins: [
		new PreTaskPlugin(),
		new webpack.DefinePlugin({
			'process.env.IS_DEV': JSON.stringify(undefined),
			'process.env.IS_DEV_IDE': JSON.stringify(undefined),
		}),
		new CopyWebpackPlugin({
			patterns: [
				{
					from: 'l10n/package.nls.json',
					to: path.resolve(__dirname, 'package.nls.json'),
					transform(content) {
						const contentStr = content.toString();
						return applyStringReplacements(contentStr);
					},
				},
				{
					from: 'l10n/package.nls.zh-cn.json',
					to: path.resolve(__dirname, 'package.nls.zh-cn.json'),
					transform(content) {
						const contentStr = content.toString();
						return applyStringReplacements(contentStr);
					},
				},
				{ from: path.resolve(__dirname, mediaSourceDir), to: path.resolve(__dirname, 'media') },
			],
		}),
	],
	devtool: false,
};
